#include "TableCommandModel.h"
#include <QModelIndex>
#include "commandlib.h"

/**
 * @brief Data() represents the getValueAt() function.
 *  It considers not only the text, but also alignment, checkability and more.
 *  See the Qt Documentation for more details.
 *  The most important thing is that an empty QVariant means "invalid", "none",
 *  or "I don't want that!". So we return it whenever we don't know any better.
 * 
 * @param index the position of the cell/data
 * @param role the kind of info the view wants
 * @return QVariant the result can take just about any time. Again QVariant() -> invalid
 */
QVariant TableCommandModel::data(const QModelIndex &index, int role = Qt::DisplayRole) const
{
    if (!index.isValid())
    {
        return QVariant();
    }
    if (role == Qt::TextAlignmentRole)
    {
        return Qt::AlignLeft;
    }
    if (role == Qt::CheckStateRole)
    {
        return QVariant();
    }

    std::shared_ptr<ICommand> icom = cL->getCommand(index.row() + 1);
    QVariant ret = QVariant();
    if (index.column() == 0)
    {
        ret = index.row()+1;
    }
    else if (index.column() == 1)
    {
        ret = icom->getName().c_str();
    }
    else if (index.column() == 2)
    {
        bool typeWasFound = false;
        std::stringstream s;

        std::shared_ptr<IGear> pG = std::dynamic_pointer_cast<IGear>(icom);
        if (pG)
        {
            typeWasFound = true;
            s << "Speed: " << pG->getSpeed();
            s << "; Duration: " << pG->getDuration();
        }

        std::shared_ptr<IDirection> pD = std::dynamic_pointer_cast<IDirection>(icom);
        if (pD)
        {
            typeWasFound = true;
            s << "Degree: " << pD->getDegree();
        }

        std::shared_ptr<IPause> pP = std::dynamic_pointer_cast<IPause>(icom);
        if (pP)
        {
            typeWasFound = true;
            s << "Duration: " << pP->getDuration();
        }

        if (typeWasFound)
            ret = s.str().c_str();
        else
            ret = QVariant();
    }
    else
    {
        ret = QVariant();
    }
    return ret;
}

/**
 * @brief The equivalent to getColumnName(int col) and getRowName(int row).
 *  returns a value depending on the input parameters.
 *  See Qt documentation for more details about Roles.
 * @param section 
 * @param orientation 
 * @param role 
 * @return QVariant columnNames from array or invalid in any other case
 */
QVariant TableCommandModel::headerData(int section, Qt::Orientation orientation, int role = Qt::DisplayRole) const
{
    if(orientation == Qt::Horizontal && role == Qt::DisplayRole)
        if (section < headerNames.size())
            return headerNames[section];
    return QVariant();
}

/**
 * @brief This function has been added because not the View initializes an update,
 *  but the Model.
 *  fireTableDataChanged() is represented with either
 *  dataChanged() if only some of the cells need to fetch an update,
 *  or layoutChanged() if the dimensions of the table have changed.
 *  These signals are emitted, they are handled outside our concerns.
 * 
 * @param _icom the Command around which the table should be updated.
 *              if nullptr an update is only triggered if the dimensions changed.
 */
void TableCommandModel::onChange(std::shared_ptr<ICommand> _icom)
{
    if (_icom != nullptr)
    {
        //update only one command (and its neighbours)
        int row = cL->getPos(_icom) - 1;
        for (int j = -1; j <= 1; ++j)
        {
            for (int i = 0; i < columnCount(); ++i)
            {
                QModelIndex index = createIndex(row + j, i, nullptr);
                emit(dataChanged(index, index));
            }
        }
    }
    int rowsNow = rowCount();
    if (rowsNow != rowsBefore)
    {
        emit(layoutChanged());
        rowsBefore = rowsNow;
    }
}
