#include "PanelConfigPause.h"
#include "Pause.h"
#include <QLayout>
#include <QFormLayout>
#include <QLabel>
#include <memory>

/**
 * @brief Construct a new Panel Config Pause:: Panel Config Pause object.
 *  Remember the ControlUI, setup Widgets and Button
 * @param _cui a pointer to the ControlUI.
 */
PanelConfigPause::PanelConfigPause(ControlUI *_cui)
{
    cui = _cui;
    setView();
    setController();
}

/**
 * @brief Fill the Panel with LineEdits for each Variable of the Command.
 *  A FormLayout is used for simplicity.
 *  To display text underneath the LineEdit an empty String is displayed
 *  on the left and an anonymous Label is displayed on the right.
 */
void PanelConfigPause::setView()
{
    QVBoxLayout *vl = new QVBoxLayout();
    setLayout(vl);
    QWidget *fw = new QWidget();
    QFormLayout *fl = new QFormLayout();
    fw->setLayout(fl);
    vl->addWidget(fw);
    bSave = new QPushButton("Save");
    vl->addWidget(bSave);

    tDuration = new QLineEdit();
    fl->addRow("Duration:", tDuration);
    fl->addRow("", new QLabel("max 8s"));
}

/**
 * @brief Connect the Button to the onSaveButton slot.
 * 
 */
void PanelConfigPause::setController()
{
    connect(bSave, SIGNAL(clicked()), this, SLOT(onSaveButton()));
}

/**
 * @brief Remember the Command, extract its values and display them in the LineEdit(s)
 * 
 * @param icom The Command to edit.
 */
void PanelConfigPause::update(std::shared_ptr<ICommand> icom)
{
    command = icom;
    std::shared_ptr<Pause> p = dynamic_pointer_cast<Pause>(command);
    if(p)
    {
        double dur = p->getDuration();
        tDuration->setText(QString::number(dur));
    }
}

/**
 * @brief Extract the numbers from the LineEdit(s)
 *  and set the according parameter in the Command.
 *  Afterwards update the ConfigPanel again to display the new correct new values
 *  and tell the ControlUI to update the Table.
 *  Here a direct call to updateTableView() is used, but it would be 
 *  more consistent with the other Panels to emit a signal and connect inside ControlUI.
 *  That way the Panel would not have to know ControlUI.
 */
void PanelConfigPause::onSaveButton()
{
    double dur = tDuration->text().toDouble();
    std::shared_ptr<Pause> p = dynamic_pointer_cast<Pause>(command);
    if(p)
    {
        if (dur < 0)
            dur = 0;
        if (dur > 8)
            dur = 8;
        p->setDuration(dur);

        this->update(command);
        cui->updateTableView(command);
    }
}