#include "PanelConfigDirection.h"
#include "Direction.h"
#include <QLayout>
#include <QFormLayout>
#include <QLabel>
#include <memory>

/**
 * @brief Construct a new Panel Config Direction:: Panel Config Direction object.
 *  Remember the ControlUI, setup Widgets and Button
 * @param _cui a pointer to the ControlUI.
 */
PanelConfigDirection::PanelConfigDirection(ControlUI *_cui)
{
    cui = _cui;
    setView();
    setController();
}

/**
 * @brief Fill the Panel with LineEdits for each Variable of the Command.
 *  A FormLayout is used for simplicity.
 *  To display text underneath the LineEdit an empty String is displayed
 *  on the left and an anonymous Label is displayed on the right.
 */
void PanelConfigDirection::setView()
{
    QVBoxLayout *vl = new QVBoxLayout();
    setLayout(vl);
    QWidget *fw = new QWidget();
    QFormLayout *fl = new QFormLayout();
    fw->setLayout(fl);
    vl->addWidget(fw);
    bSave = new QPushButton("Save");
    vl->addWidget(bSave);

    tDegree = new QLineEdit();
    fl->addRow("Degree:", tDegree);
    fl->addRow("", new QLabel("-90 to +90 degree"));
}

/**
 * @brief Connect the Button to the onSaveButton slot.
 * 
 */
void PanelConfigDirection::setController()
{
    connect(bSave, SIGNAL(clicked()), this, SLOT(onSaveButton()));
}

/**
 * @brief Remember the Command, extract its values and display them in the LineEdit(s)
 * 
 * @param icom The Command to edit.
 */
void PanelConfigDirection::update(std::shared_ptr<ICommand> icom)
{
    command = icom;
    std::shared_ptr<Direction> d = dynamic_pointer_cast<Direction>(command);
    if(d)
    {
        int deg = d->getDegree();
        tDegree->setText(QString::number(deg));
    }
}

/**
 * @brief Extract the numbers from the LineEdit(s)
 *  and set the according parameter in the Command.
 *  Afterwards update the ConfigPanel again to display the new correct new values
 *  and tell the ControlUI to update the Table.
 *  Here a direct call to updateTableView() is used, but it would be 
 *  more consistent with the other Panels to emit a signal and connect inside ControlUI.
 *  That way the Panel would not have to know ControlUI.
 */
void PanelConfigDirection::onSaveButton()
{
    int deg = tDegree->text().toInt();
    std::shared_ptr<Direction> d = dynamic_pointer_cast<Direction>(command);
    if(d)
    {
        d->setDegree(deg);

        this->update(command);
        cui->updateTableView(command);
    }
}