#include "PanelCommandTable.h"
#include <QBoxLayout>
#include <QHeaderView>

/**
 * @brief Construct a new Panel Command Table:: Panel Command Table object
 *  Remember the ControlModel, set Widgets and connect Buttons.
 * @param _cM Pointer to the ControlModel
 */
PanelCommandTable::PanelCommandTable(ControlModel *_cM)
{
    cM = _cM;
    setView();
    setController();
}

/**
 * @brief Fill the Widgets and parametrize them.
 *  Also get the SelectionModel and udate the Rover Label
 */
void PanelCommandTable::setView()
{
    QBoxLayout *vLayout = new QBoxLayout(QBoxLayout::Direction::TopToBottom);
    setLayout(vLayout);

    QWidget *buttons = new QWidget();
    QBoxLayout *hLayout = new QBoxLayout(QBoxLayout::Direction::LeftToRight);
    buttons->setLayout(hLayout);

    tCommands = new QTableView();
    tM = new TableCommandModel(&cM->getCommandList());
    tCommands->setModel(tM);
    tCommands->setSelectionBehavior(QAbstractItemView::SelectionBehavior::SelectRows);
    tCommands->setSelectionMode(QAbstractItemView::SelectionMode::SingleSelection);
    tCommands->setWordWrap(false);
    tCommands->horizontalHeader()->setStretchLastSection(true);
    tCommands->verticalHeader()->hide();
    vLayout->addWidget(tCommands);
    vLayout->addWidget(buttons);

    bRemove = new QPushButton("Remove");
    hLayout->addWidget(bRemove);
    bUp = new QPushButton("Up");
    hLayout->addWidget(bUp);
    bDown = new QPushButton("Down");
    hLayout->addWidget(bDown);

    hLayout->addStretch();
    lRover = new QLabel();
    hLayout->addWidget(lRover);
    bStart = new QPushButton("Start");
    hLayout->addWidget(bStart);
    bStop = new QPushButton("Stop");
    hLayout->addWidget(bStop);

    lSM = tCommands->selectionModel();
    updateSelectedRover();
}

/**
 * @brief Connect the Buttons to the respective slots.
 *  Connect currentRowChanged() of the SelectionModel to the respective slot.
 */
void PanelCommandTable::setController()
{
    connect(bRemove, SIGNAL(clicked()), this, SLOT(onRemoveButton()));
    connect(bUp,     SIGNAL(clicked()), this, SLOT(onUpButton()));
    connect(bDown,   SIGNAL(clicked()), this, SLOT(onDownButton()));
    connect(bStart,  SIGNAL(clicked()), this, SLOT(onStartButton()));
    connect(bStop,   SIGNAL(clicked()), this, SLOT(onStopButton()));
    connect(lSM,     SIGNAL(currentRowChanged(const QModelIndex, const QModelIndex)), this, SLOT(onSelectionChange(const QModelIndex, const QModelIndex)));
}

/**
 * @brief First the currently selected row is fetched.
 *  Second the selection is then reset in case the selected index
 *  would no more be valid after the removal.
 *  Third CommandList.remove() is called, the returned ICommand is not saved.
 *  Fourth a reasonable new selection is searched.
 *  Select the first or last element if those were removed
 *  or select the same row as was already selected.
 *  The selection is done updateTable().
 */
void PanelCommandTable::onRemoveButton()
{
    int index = lSM->currentIndex().row();
    if (index >= 0) //valid selection
    {
        lSM->reset();
        cM->getCommandList().remove(index + 1);
        std::shared_ptr<ICommand> newSelect;
        if (index == 0) // first item removed
            newSelect = cM->getCommandList().getCommand(1);
        else if (index == cM->getCommandList().getSize() - 1) //last item removed
            newSelect = cM->getCommandList().getCommand(cM->getCommandList().getSize());
        else // middle item removed
            newSelect = cM->getCommandList().getCommand(index);
        emit(pleaseUpdateTable(newSelect));
    }
}

/**
 * @brief Get the current selection, if it is valid call CommandList.moveUp()
 *  and initialize a tableUpdate by emitting pleaseUpdateTable() with the changed Command.
 * 
 */
void PanelCommandTable::onUpButton()
{
    int index = lSM->currentIndex().row();
    if (index >= 0) //valid selection
    {
        std::shared_ptr<ICommand> icom = cM->getCommandList().moveUp(index + 1);
        emit(pleaseUpdateTable(icom));
    }
}

/**
 * @brief Get the current selection, if it is valid call CommandList.moveDown()
 *  and initialize a tableUpdate by emitting pleaseUpdateTable() with the changed Command.
 * 
 */
void PanelCommandTable::onDownButton()
{
    int index = lSM->currentIndex().row();
    if (index >= 0) //valid selection
    {
        std::shared_ptr<ICommand> icom = cM->getCommandList().moveDown(index + 1);
        emit(pleaseUpdateTable(icom));
    }
}

/**
 * @brief Tell the ControlModel to start the execution.
 * 
 */
void PanelCommandTable::onStartButton()
{
    cM->start();
}

/**
 * @brief Tell the ControlModel to stop the execution.
 * 
 */
void PanelCommandTable::onStopButton()
{
    cM->stop();
}

/**
 * @brief Tell the tableModel to inform it's listeners of updated data.
 *  Afterwards check set the either set the selection to the changed Command
 *  or remove it if no specific Command was changed.
 *  This also happend when a command is tried to be moved out of bounds of the list
 *  or the last item is removed.
 * 
 * @param _icom the changed Command that should be selected.
 */
void PanelCommandTable::updateTable(std::shared_ptr<ICommand> _icom)
{
    tM->onChange(_icom);

    int row = cM->getCommandList().getPos(_icom) - 1;
    QModelIndex index;
    if (_icom)
        index = tM->index(row, 0);
    else
        index = tM->index(- 1, 0);
    lSM->setCurrentIndex(index, QItemSelectionModel::SelectionFlag::ClearAndSelect | QItemSelectionModel::SelectionFlag::Rows);
}

/**
 * @brief Get the selected Rover from the ControlModel.
 *  Change the roverLabel according to the result.
 *  With optional.value_or() this could have been a neat one-liner.
 */
void PanelCommandTable::updateSelectedRover()
{
    std::optional<std::string> roverId = cM->getSelectedRoverId();
    if (roverId.has_value())
    {
        QString s = "Rover: ";
        QString id = roverId.value().c_str();
        lRover->setText(s + id);
    }
    else
        lRover->setText("Rover: - ");
}

/**
 * @brief From the currently selected row get the ICommand
 *  and initialize the update of the configPanel.
 * 
 * @param current currently selected index
 * @param previous previously selected index
 */
void PanelCommandTable::onSelectionChange(const QModelIndex &current, const QModelIndex &previous)
{
    int index = current.row() + 1;
    std::shared_ptr<ICommand> icom = cM->getCommandList().getCommand(index);
    emit(pleaseUpdateConfig(icom));
}