#include <QApplication>
#include <QSplitter>
#include <QFileDialog>
#include <QDir>
#include <memory>
#include "commandlib.h"
#include "ControlUI.h"
#include "PanelConfigDirection.h"
#include "PanelConfigPause.h"
#include "PanelConfigGear.h"
#include "PanelConfigDefault.h"
#include "RoverDialog.h"
#include "AboutDialog.h"

/**
 * @brief Construct a new ControlUI:: ControlUI object.
 *  Fill the array of configPanels, set View, control and add a Menubar.
 * @param parent An overlying widget, if it would exist. In our case none.
 */
ControlUI::ControlUI(QWidget *parent)
    : QMainWindow(parent)
{
    panelConfigs =
        {
            new PanelConfigDefault(this),
            new PanelConfigDirection(this),
            new PanelConfigGear(this),
            new PanelConfigPause(this)
        };
    setView();
    addMenuBar();
    setController();
}

/**
 * @brief Fill the MainWindow with Widgets.
 * 
 */
void ControlUI::setView()
{
    setWindowTitle("ControlDeveloper");
    QSplitter *splitterV = new QSplitter(Qt::Vertical);
    setCentralWidget(splitterV);
    QSplitter *splitterH = new QSplitter(Qt::Horizontal);
    splitterV->addWidget(splitterH);

    messageArea = new QTextEdit();
    messageArea->setReadOnly(true);
    messageArea->setFontPointSize(11);
    splitterV->addWidget(messageArea);

    panelTypes = new PanelCommandTypes(&ControlModel::getInstance());
    splitterH->addWidget(panelTypes);
    panelTable = new PanelCommandTable(&ControlModel::getInstance());
    splitterH->addWidget(panelTable);
    configStack = new QStackedWidget();
    splitterH->addWidget(configStack);
    for (auto &&i : panelConfigs)
    {
        configStack->addWidget(i);
    }
}

/**
 * @brief Add a MenuBar to the MainWindow.
 *  The Items in the Menus are QActions which could also be
 *  used in a QToolbar.
 * 
 * @attention This Function has been renamed to AddMenuBar
 *  because the MainWindow already has a setMenuBar() function
 *  to add one to the GUI.
 */
void ControlUI::addMenuBar()
{
    menuBar = new QMenuBar();
    setMenuBar(menuBar);
    mFile = new QMenu("File");
    menuBar->addMenu(mFile);
    aOpen = new QAction("Open...", this);
    mFile->addAction(aOpen);
    aSave = new QAction("Save...", this);
    mFile->addAction(aSave);
    aRover = new QAction("Select Rover...", this);
    mFile->addAction(aRover);
    aExit = new QAction("Exit", this);
    mFile->addAction(aExit);
    mHelp = new QMenu("Help");
    menuBar->addMenu(mHelp);
    aAbout = new QAction("About...", this);
    mHelp->addAction(aAbout);
}

/**
 * @brief Connect the Buttons with the according slots,
 *  subscribe to the ControlModel and
 *  and connect the Panels update signals to updateConfigView().
 */
void ControlUI::setController()
{
    ControlModel::getInstance().addControlModelListener(this);
    connect(aOpen, SIGNAL(triggered()), this, SLOT(onOpen()));
    connect(aSave, SIGNAL(triggered()), this, SLOT(onSave()));
    connect(aRover, SIGNAL(triggered()), this, SLOT(onRover()));
    connect(aExit, SIGNAL(triggered()), this, SLOT(onExit()));
    connect(aAbout, SIGNAL(triggered()), this, SLOT(onAbout()));
    connect(panelTable, SIGNAL(pleaseUpdateTable(std::shared_ptr<ICommand>)), this, SLOT(updateTableView(std::shared_ptr<ICommand>)));
    connect(panelTypes, SIGNAL(pleaseUpdateTable(std::shared_ptr<ICommand>)), this, SLOT(updateTableView(std::shared_ptr<ICommand>)));
    connect(panelTable, SIGNAL(pleaseUpdateConfig(std::shared_ptr<ICommand>)), this, SLOT(updateConfigView(std::shared_ptr<ICommand>)));
}

/**
 * @brief The central starting point for table updates from anywhere in the GUI.
 *  Simply tells the tablePanel to update the Table.
 *  This Function has been declared a slot so that it can be connected 
 *  to the pleaseUpdateTable() signals from other Panels.
 *  The function can still be called directly.
 * @param _icom The Command which to update.
 */
void ControlUI::updateTableView(std::shared_ptr<ICommand> _icom)
{
    panelTable->updateTable(_icom);
}

/**
 * @brief Used to set the configPanel to the right type and Command.
 *  This Function has been declared a slot so that it can be connected 
 *  to the pleaseUpdateConfig() signals from other Panels.
 *  The function can still be called directly.
 * @param _icom The Command which will be edited in the configPanel.
 */
void ControlUI::updateConfigView(std::shared_ptr<ICommand> _icom)
{
    if (dynamic_pointer_cast<IGear>(_icom))
        configStack->setCurrentIndex(confNum::cGear);
    else if (dynamic_pointer_cast<IPause>(_icom))
        configStack->setCurrentIndex(confNum::cPause);
    else if (dynamic_pointer_cast<IDirection>(_icom))
        configStack->setCurrentIndex(confNum::cDirection);
    else
        configStack->setCurrentIndex(confNum::cDefault);

    static_cast<PanelCommandConfig *>(configStack->currentWidget())->update(_icom);
}

/**
 * @brief A function of the IControlModelListener interface.
 *  Appends the message to the TextEdit.
 *  append() adds the string in a new Line.
 * @param _message the message to be displayed.
 */
void ControlUI::messageUpdated(std::string _message)
{
    messageArea->append(_message.c_str());
}

/**
 * @brief A function of the IControlModelListener interface.
 *  Calls messageUpdated() to inform of the rover selection.
 *  Tells the tablePanel to update the selected rover field.
 */
void ControlUI::roverUpdated()
{
    panelTable->updateSelectedRover();
    messageUpdated("A Rover has been selected.");
}

/**
 * @brief Called from a QAction in the Menubar.
 *  Uses a QFileDialog to get a filepath,
 *  if successful it calls ControlModel.readCommands()
 *  and calls updateTableView with no specified Command.
 */
void ControlUI::onOpen()
{
    QString filter = "Text Files (*.txt)";
    QString qpath = QFileDialog::getOpenFileName(this, "Read a File", QDir::homePath(), filter);
    if (!qpath.isEmpty())
    {
        ControlModel::getInstance().readCommands(qpath.toStdString());
        updateTableView(nullptr);
    }
}

/**
 * @brief Called from a QAction in the Menubar.
 *  Uses a QFileDialog to get a filepath,
 *  if successful calls ControlModel.writeCommands()
 *  to save the list to a file.
 */
void ControlUI::onSave()
{
    QString filter = "Text Files (*.txt)";
    QString qpath = QFileDialog::getSaveFileName(this, "Save to File", QDir::homePath(), filter);
    std::cout << qpath.toStdString() << std::endl;
    if (!qpath.isEmpty())
    {
        ControlModel::getInstance().writeCommands(qpath.toStdString());
    }
}

/**
 * @brief Called from a QAction in the Menubar.
 *  Opens the RoverDialog which will select a Rover.
 *  the dialog is displayed with exec() rather than show().
 *  This way the Dialog is modal, meaning the rest pf the program
 *  blocks until it is closed.
 */
void ControlUI::onRover()
{
    RoverDialog r(&ControlModel::getInstance());
    r.exec();
}

/**
 * @brief Called from a QAction in the Menubar.
 *  Simply calls QApplication::exit() to exit the application.
 */
void ControlUI::onExit()
{
    QApplication::exit();
}

/**
 * @brief Called from a QAction in the Menubar.
 *  Opens the AboutDialog to display information about the program.
 *  The dialog is displayed with exec() rather than show().
 *  This way the Dialog is modal, meaning the rest pf the program
 *  blocks until it is closed.
 */
void ControlUI::onAbout()
{
    AboutDialog a;
    a.exec();
}

/**
 * @brief Destroy the ControlUI:: ControlUI object
 *  Because QActions can be represented by multiple elements of the GUI
 *  e.g. MenuBar and ToolBar none of them can reliably take ownership of the Actions.
 *  Bacause of this we delete them ourselves.
 *  Likely this doesn't matter much as the ControlUI will live as long as the program
 *  and the memory is cleared by the OS after that anyway.
 */
ControlUI::~ControlUI()
{
    delete aOpen;
    delete aSave;
    delete aRover;
    delete aExit;
    delete aAbout;
}