#include "ControlModel.h"


/**
 * @brief Construct a new Control Model:: Control Model object
 *  Registers itself to ComHandler
 */
ControlModel::ControlModel()
{
    ComHandler::getInstance().registerComListener(this);
}

/**
 * @brief Destroy the Control Model:: Control Model object
 *  Unregisters itself from ComHandler
 */
ControlModel::~ControlModel()
{
    ComHandler::getInstance().unregisterComListener(this);
}

/**
 * @brief As ControlModel is Singleton, this function manages
 *  the static instance and returns a reference to it.
 * @return ControlModel& the Instance of ControlModel
 */
ControlModel& ControlModel::getInstance()
{
    static ControlModel instance = ControlModel();
    return instance;
}

/**
 * @brief Will notify all IControlModelListeners subscribed to ControlModel
 *  with a string corresponding to the given parameters.
 * @param _command an optional ICommand. The Message will be getConfig() or "no Command"
 * @param _state The ComState the ComHandler is in.
 */
void ControlModel::commandPerformed(std::optional<std::shared_ptr<ICommand>> _command, ComState _state)
{
    std::string commandPart = "";
    std::shared_ptr<Command> command = std::dynamic_pointer_cast<Command>(_command.value_or(nullptr));
    if (command)
        commandPart = command->getConfig();
    else
        commandPart = "no Command";

    std::string statePart = "";
    switch (_state)
    {
    case ComState::connection_error:
        statePart = "Connection Error";
        break;
    case ComState::rover_finish_ICommand:
        statePart = "Rover finish ICommand";
        break;
    case ComState::rover_in_use:
        statePart = "Rover in Use";
        break;
    case ComState::rover_running_ICommand:
        statePart = "Rover running ICommand";
        break;
    case ComState::send_ICommand:
        statePart = "Send ICommand";
        break;
    default:
        statePart = "";
        break;
    }
    notifyMessageChanged("Command: " + commandPart + " - State: " + statePart);
}

/**
 * @brief Will gather every ICommand from the CommandList into a vector and
 *  hand it over to the ComHandler together with the selected Rover to start the
 *  execution.
 * @return the result of ComHandler.start()
 */
bool ControlModel::start()
{
    std::vector<std::shared_ptr<ICommand>> commands;
    uint size = list.getSize();
    //0 is root element
    // iterator would help greatly with of by one errors
    for (int i = 1; i <= size; ++i) 
    {
        commands.push_back(list.getCommand(i));
    }

    return ComHandler::getInstance().start(commands, selectedRover);
}

/**
 * @brief Will tell the ComHandler to stop the execution of commands
 * 
 * @return the result of ComHandler.stop()
 */
bool ControlModel::stop()
{
    return ComHandler::getInstance().stop();
}

/**
 * @brief Uses ObjectFileHandler to parse a file for Commands 
 *  and adds them to the CommandList
 * @param _fileName relative Path to the file to be read.
 * @see ObjectFileHandler
 */
void ControlModel::readCommands(std::string _fileName)
{
    ObjectFileHandler fH = ObjectFileHandler(_fileName);
    std::vector<std::shared_ptr<ICommand>> v;
    fH.read(v);
    for (auto &&i : v)
    {
        list.add(i);
    }
}

/**
 * @brief Gathers all Commands from the CommandList, then uses
 *  ObjectFileHandler to write them to a file
 * @param _fileName relative Path to the file to be written to. (See ObjectFileHandler)
 * @see ObjectFileHandler
 */
void ControlModel::writeCommands(std::string _fileName)
{
    std::vector<std::shared_ptr<ICommand>> v;
    uint size = list.getSize();
    // 0 is root element
    // iterator would help greatly with of by one errors
    for (int i = 1; i <= size; ++i) 
    {
        v.push_back(list.getCommand(i));
    }
    ObjectFileHandler fH = ObjectFileHandler(_fileName);
    fH.write(v);
}

/**
 * @brief Gets free Rovers from RoverHandler and sets the first
 *  one as the selectedRover.
 *  If the List is empty, e.g. during Simulation, an empty Rover is created.
 *  All subsrcribers to ControlModel are notified of the rover change.
 */
void ControlModel::setSelectedRover()
{
    std::vector<Rover> rovers = RoverHandler::getFreeRover();
    if (!rovers.empty())
    {
        selectedRover = rovers.front();
        notifyRoverChanged();
    }
    else
    {
        selectedRover = Rover(); //some example Rover
        notifyRoverChanged();
    }
}

/**
 * @brief Since it is possible no rover is selected the value is optional.
 * 
 * @return std::optional<std::string> the Id of the Rover or nullopt if no Rover was selected.
 */
std::optional<std::string> ControlModel::getSelectedRoverId() 
{
if (selectedRover)
    return selectedRover.value().getId();
else
    return std::nullopt;
}
