#include "CommandListOWN.h"

/**
 * @brief Returns the number of Elements in the List
 * 
 * @return int Elements in the List
 */
int CommandList::getSize()
{
    std::shared_ptr<Element> ptr = root;
    int i = 0;
    for (; ptr->getNext(); ++i)
    {
        ptr = ptr->getNext();
    }
    return i;
}

/**
 * @brief Removes every Element from the List.
 * 
 */
void CommandList::clear()
{
    std::shared_ptr<Element> current = root;
    std::shared_ptr<Element> next = current->getNext();

    while (next)
    {
        current->setPrev(nullptr);
        current->setNext(nullptr);
        current = next;
        next = next->getNext();
    }
}

/**
 * @brief Adds a Command of Type to the end of the list.
 * This template does not check if the given Type inherits from Command.
 * @tparam Type implicit
 * @param _cmd any Command that inherits from ICommand
 * @return std::shared_ptr<ICommand> to the newly instanciated Command
 */
template <typename Type>
std::shared_ptr<ICommand> CommandList::add(const Type _cmd)
{
    std::shared_ptr<Element> newElement = std::make_shared<Element>(_cmd);
    std::shared_ptr<Element> end = getElement(getSize());
    newElement->setPrev(end);
    end->setNext(newElement);
    return newElement->getCommand();
}

/**
 * @brief Clears list before adding 5 Commands to the list.
 * 
 */
void CommandList::createCommands()
{
    clear();

    add(Gear(10, 2.71));
    add(Direction(90));
    add(Pause(2.5));
    add(Direction(-90));
    add(Gear(-10, 2.71));
}

/**
 * @brief Prints out every Command in the list to std::cout using the getConfig() method.
 * 
 */
void CommandList::printCommands()
{
    std::shared_ptr<Element> current = root;
    while (current->getNext())
    {
        //In this order to not print out the root element.
        current = current->getNext();
        std::shared_ptr<Command> command = std::dynamic_pointer_cast<Command>(current->getCommand());
        if (command != nullptr)
            std::cout << command->getConfig() << '\n';
    }
}

/**
 * @brief Removes the Element at _pos from the list.
 * 
 * @param _pos of the Element to be deleted
 * @return std::shared_ptr<ICommand> to the Command the Element was holding
 * or nullptr, if unsuccessful
 */
std::shared_ptr<ICommand> CommandList::remove(unsigned int _pos)
{
    std::shared_ptr<Element> toRemove = getElement(_pos);
    if (toRemove == nullptr | toRemove == root)
    {
        return nullptr;
    }
    std::shared_ptr<Element> prev = toRemove->getPrev();
    std::shared_ptr<Element> next = toRemove->getNext();
    prev->setNext(next);
    if (next)
        next->setPrev(prev);

    return toRemove->getCommand();
}

/**
 * @brief Get the Element at _pos of the list.
 * 
 * @param _pos Position of the wanted Element.
 * @return std::shared_ptr<Element> to the Element at _pos or nullptr if unsuccessful.
 */
std::shared_ptr<Element> CommandList::getElement(unsigned int _pos)
{
    if (_pos > getSize())
    {
        return nullptr;
    }
    std::shared_ptr<Element> elem = root;
    for (int i = 0; elem->getNext() && i < _pos; ++i)
    {
        elem = elem->getNext();
    }
    return elem;
}

/**
 * @brief Get the Command at _pos of the list utilizing CommandList::getElement.
 * 
 * @param _pos of the wanted Command
 * @return std::shared_ptr<ICommand> to the wanted Command or nullptr if unsuccessful.
 */
std::shared_ptr<ICommand> CommandList::getCommand(unsigned int _pos)
{
    std::shared_ptr<Element> elem = getElement(_pos);
    if (elem == nullptr)
    {
        return nullptr;
    }
    else
    {
        return elem->getCommand();
    }
}

/**
 * @brief Searches the list for a std::shared_ptr<ICommand>.
 * 
 * @param _cmd to look up.
 * @return int position of the std::shared_ptr<ICommand> or -1 if unsuccessful.
 */
int CommandList::getPos(std::shared_ptr<ICommand> _cmd)
{
    std::shared_ptr<Element> elem = root;
    int i = 0;
    int size = getSize();
    while (elem->getNext() && elem->getCommand() != _cmd)
    {
        elem = elem->getNext();
        ++i;
        if (i > size)
        {
            return -1;
        }
    }
    return i;
}

/**
 * @brief Moves up an Element in the list.
 * Switching the Element at _pos with the Element at _pos-1.
 * @param _pos of the Element to move up
 * @return std::shared_ptr<ICommand> held by the moved Element or nullptr if unsuccessful.
 */
std::shared_ptr<ICommand> CommandList::moveUp(unsigned int _pos)
{
    if (_pos <= 1)
    {
        return nullptr;
    }
    std::shared_ptr<Element> ptr = getElement(_pos);
    std::shared_ptr<Element> prevPtr = getElement(_pos -1);
    if (ptr == nullptr | prevPtr == nullptr)
    {
        return nullptr;
    }
    std::shared_ptr<ICommand> toReturn = ptr->getCommand();

    prevPtr->getPrev()->setNext(ptr);
    ptr->setPrev(prevPtr->getPrev());

    if (ptr->getNext() != nullptr) // -> not last element
    {
        ptr->getNext()->setPrev(prevPtr);
    }
    prevPtr->setNext(ptr->getNext());

    prevPtr->setPrev(ptr);
    ptr->setNext(prevPtr);

    return toReturn;
}

/**
 * @brief Moves down an Element in the list.
 * Switching the Element at _pos with the Element at _pos+1.
 * Probably one of these functions could use the other one with an offset, but whatever.
 * @param _pos of the Element to move down
 * @return std::shared_ptr<ICommand> held by the moved Element or nullptr if unsuccessful.
 */
std::shared_ptr<ICommand> CommandList::moveDown(unsigned int _pos)
{
    std::shared_ptr<Element> ptr = getElement(_pos);
    std::shared_ptr<Element> nextPtr = getElement(_pos +1);
    if (ptr == nullptr | nextPtr == nullptr | ptr == root)
    {
        return nullptr;
    }
    std::shared_ptr<ICommand> toReturn = ptr->getCommand();

    ptr->getPrev()->setNext(nextPtr);
    nextPtr->setPrev(ptr->getPrev());

    if (nextPtr->getNext() != nullptr) // -> not pre last element
    {
        nextPtr->getNext()->setPrev(ptr);
    }
    ptr->setNext(nextPtr->getNext());

    ptr->setPrev(nextPtr);
    nextPtr->setNext(ptr);

    return toReturn;
}