#include "ObjectFileHandler.h"

/**
 * @brief Construct a new ObjectFileHandler, no file manipulation done here.
 * 
 * @param _file The relative filepath to the used file.
 */
ObjectFileHandler::ObjectFileHandler(std::string _file)
{
    fileName = _file;
}

/**
 * @brief Write out the ICommands to the file specified in the constructor.
 * 
 * @param source Reference of a vector filled with ICommands to be saved.
 * @returns false if something went wrong, true on success.
 */
bool ObjectFileHandler::write(const std::vector <std::shared_ptr<ICommand>> &source)
{
    std::fstream file;
    file.open(fileName, file.out);
    if (!file.is_open())
        return false;

    for (std::shared_ptr<ICommand> i : source)
    {
        bool typeWasFound = false;
        std::stringstream s;

        std::shared_ptr<IGear> pG = std::dynamic_pointer_cast<IGear>(i);
        if (pG)
        {
            typeWasFound = true;
            s << IGear::gear << ' ';
            s << pG->getSpeed() << ' ';
            s << pG->getDuration();
        }

        std::shared_ptr<IDirection> pD = std::dynamic_pointer_cast<IDirection>(i);
        if (pD)
        {
            typeWasFound = true;
            s << IDirection::direction << ' ';
            s << pD->getDegree();
        }

        std::shared_ptr<IPause> pP = std::dynamic_pointer_cast<IPause>(i);
        if (pP)
        {
            typeWasFound = true;
            s << IPause::pause << ' ';
            s << pP->getDuration();
        }

        if (!typeWasFound)
            return false;
        s << '\n';
        file << s.rdbuf();
    }
    file << std::flush;
    file.close();
    return true;
}

/**
 * @brief Reads the file specified in the constructor, interpreting the file
 *  and appending matching ICommands to the given vector.
 * @param destination The vector to which to append the ICommands from the file.
 * @return false if something went wrong, true on success.
 */
bool ObjectFileHandler::read(std::vector <std::shared_ptr<ICommand>> &destination)
{
    std::fstream file;
    file.open(fileName, file.in);
    if (!file.is_open())
        return false;

    std::string name;
    while (file >> name)
    {
        if (name == IGear::gear)
        {
            int speed;
            file >> speed;
            double duration;
            file >> duration;
            destination.emplace_back(std::make_shared<Gear>(speed, duration));
        }
        else if (name == IDirection::direction)
        {
            int degree;
            file >> degree;
            destination.emplace_back(std::make_shared<Direction>(degree));
        }
        else if (name == IPause::pause)
        {
            double duration;
            file >> duration;
            destination.emplace_back(std::make_shared<Pause>(duration));
        }
        else
        {
            //This shouldn't happen.
            return false;
        }
    }

    file.close();
    return true;
}