#include "ComHandler.h"
#include <iostream>

/**
 * @brief Manages the static Singleton instance of ComHandler
 * 
 * @return ComHandler& the Instance of ComHandler
 */
ComHandler& ComHandler::getInstance()
{
    static ComHandler instance = ComHandler();
    return instance;
}

/**
 * @brief Registers an IComListener to ComHandler
 *  for use in an observer pattern.
 * @param cl the raw pointer to be registered.
 */
void ComHandler::registerComListener(IComListener *cl)
{
    comListener = cl;
}

/**
 * @brief Unregister an IComListener from ComHandler.
 *  Unregisters the Listener only if the pointer matches
 *  the one already registered to ComHandler.
 * @param cl the raw pointer to the Listener
 */
void ComHandler::unregisterComListener(IComListener *cl)
{
    if (comListener == cl)
        comListener.reset();
}

/**
 * @brief Will currently always simulate the execution as the rover communitaction is not implemented.
 * 
 * @param commands A vector of ICommands to be executed
 * @param rover The rover to be used (currently ignored!)
 * @return true if a Rover was given -> real execution
 * @return false if no Rover was given -> simulation
 */
bool ComHandler::start(std::vector<std::shared_ptr<ICommand>> commands, std::optional<Rover> rover)
{
    if (rover)
    {
        //* Funtionality unknown
        std::cout << "Many robot, such wow!\nSimuliere stattdessen:\n";
        start(commands, std::nullopt);
    }
    else
    {
        std::cout << "Simuliere Ausführung durch Listeneraufrufe.\n";
        for (auto &&i : commands)
        {
            if (comListener)
                comListener.value()->commandPerformed(i, ComState::connection_error);
        }
    }
    return rover ? true : false;
}